/**
 * 
 */
package gov.va.med.mhv.usermgmt.integration.service.impl;

import gov.va.med.mhv.core.messages.MessagesStringBuilder;
import gov.va.med.mhv.core.util.PatientCorrelationStatusUtils;
import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.core.util.ServiceResponseUtils;
import gov.va.med.mhv.integration.phr.service.ServiceResponse;
import gov.va.med.mhv.integration.registry.transfer.Facility;
import gov.va.med.mhv.integration.registry.transfer.Patient;
import gov.va.med.mhv.integration.registry.transfer.PersonalInfo;
import gov.va.med.mhv.integration.registry.transfer.Status;
import gov.va.med.mhv.integration.registry.transfer.StatusType;
import gov.va.med.mhv.usermgmt.bizobj.FacilityInfoBO;
import gov.va.med.mhv.usermgmt.bizobj.UserProfileBO;
import gov.va.med.mhv.usermgmt.common.MessagesSet;
import gov.va.med.mhv.usermgmt.common.ServiceFactory;
import gov.va.med.mhv.usermgmt.enumeration.ActivityActorTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.AuthenticationStatus;
import gov.va.med.mhv.usermgmt.enumeration.PatientCorrelationStatus;
import gov.va.med.mhv.usermgmt.integration.service.PatientIdentityService;
import gov.va.med.mhv.usermgmt.service.PatientServiceResponse;
import gov.va.med.mhv.usermgmt.service.UserProfileService;
import gov.va.med.mhv.usermgmt.transfer.InPersonAuthentication;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;
import gov.va.med.mhv.usermgmt.transfer.UserProfileDeactivationReason;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


/**
 * Adapts the integration PatientIdentityService (EJB) interface to the 
 * internal business service interface (POJO).
 * The PatientIdentityService uses an interface with service
 * responses, just like the generated Atlas services, 
 * whereas the PatientIdentityBean defines its own.
 * This adapter converts between the two different interfaces.
 * The adapter translates the messages from the Atlas like service to
 * the interfaces of the EJB.
 * @author PII
 *
 */
public class PatientIdentityServiceImpl implements PatientIdentityService {
    

    private static final Log LOG = LogFactory.getLog(
        PatientIdentityServiceImpl.class);
    
    /* (non-Javadoc)
     * @see gov.va.med.mhv.integration.registry.services.ejb.
     * PatientIdentityServiceRemote#getPatientForICN(java.lang.String)
     */
    public Patient getPatientForICN(String icn) {
        Precondition.assertNotBlank("icn", icn);
        PatientServiceResponse response = getPatientByIcn(icn);
        return (response.getPatient() != null)
            ? RegistryInterfaceUtils.toPatient(response.getPatient()) 
            : null;
    }

    /* (non-Javadoc)
     * @see gov.va.med.mhv.integration.registry.services.ejb.
     * PatientIdentityServiceRemote#finalizeCorrelation(
     * gov.va.med.mhv.integration.registry.transfer.Patient, 
     * gov.va.med.mhv.integration.registry.transfer.Status)
     */
    public Status finalizeCorrelation(Patient patient, Status status) {
        Precondition.assertNotNull("patient", patient);
        final String operation = "Finalize correlation"; 
        if (LOG.isDebugEnabled()) {
            LOG.debug(operation + " of " + describe(patient) + 
                " where MPI status is " + describe(status));
        }
        PatientServiceResponse response = getPatient(patient);
        if (response.getPatient() == null) {
            status = createPatientDoesNotExistStatus();
        } else if (!PatientCorrelationStatusUtils.isPendingCorrelation(
                        response.getPatient())) 
        {
            status = createPatientNotPendingCorrelationResponse();
        } else {
            String error = getError(status);
            // store if patient was pending synchronization, because 
            // finalizeCorrelation will remove this record
            boolean wasPendingSynchronization = isPendingSynchronization(
                response.getPatient()); 
            handleMessages(getPatientService().finalizeCorrelation(response.
                getPatient(), error));
            status = finalizeAuthentication(response.getPatient(), error, 
                wasPendingSynchronization);
        }
        logStatus(operation, patient, status);
        return status;
    }

    private Status finalizeAuthentication(
        gov.va.med.mhv.usermgmt.transfer.Patient patient, String error,
        boolean wasPendingSynchronization) 
    {
        InPersonAuthentication ipa = getIPA(patient);
        if (ipa == null) {
            return createPatientIsNotPendingAuthenticationStatus();
        }
        if (InPersonAuthenticationStatusUtils.isAuthenticated(ipa)) {
            // NOTE: This scenario should eventually go away,
            // when all patient authenticated before introduction
            // of the MPI synchronization feature are correlated 
            if (!wasPendingSynchronization) {
                return createPatientIsNotPendingSynchronizationStatus();
            } // else: Nothing else to do; already authenticated
        } else if (!InPersonAuthenticationStatusUtils.
                isPendingAuthentication(ipa)) 
        {
            return createPatientIsNotPendingAuthenticationStatus();
        } else {
            handleMessages(getIPAService().finalizeAuthentication(ipa, error));
        }
        return RegistryInterfaceUtils.createOkStatus();
    }
    
    private InPersonAuthentication getIPA(
        gov.va.med.mhv.usermgmt.transfer.Patient patient)
    {
        InPersonAuthenticationService ipaService = getIPAService();
        InPersonAuthenticationServiceResponse ipaServiceResponse = 
            ipaService.getAuthenticationForPatient(patient.getId());
        handleMessages(ipaServiceResponse);
        return ipaServiceResponse.getInPersonAuthentication(); 
    }
    
    private boolean isPendingSynchronization(
        gov.va.med.mhv.usermgmt.transfer.Patient patient) 
    {
        return (patient != null)
            && (patient.getPatientSynchronizations() != null) 
            && !patient.getPatientSynchronizations().isEmpty();
    }
    
    /* (non-Javadoc)
     * @see gov.va.med.mhv.integration.registry.services.ejb.
     * PatientIdentityServiceRemote#finalizeUncorrelation(
     * gov.va.med.mhv.integration.registry.transfer.Patient, 
     * gov.va.med.mhv.integration.registry.transfer.Status)
     */
    public Status finalizeUncorrelation(Patient patient, Status status) {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotEmpty("patient.getIcn()", patient.getIcn());
        final String operation = "Finalize uncorrelation"; 
        if (LOG.isDebugEnabled()) {
            LOG.debug(operation + " of " + describe(patient) 
                + " where MPI status is " +  describe(status));
        }
        PatientServiceResponse response = getPatientByIcn(patient.getIcn());
        if (response.getPatient() == null) {
            status = createPatientDoesNotExistStatus();
        } else if (!PatientCorrelationStatusUtils.isPendingUncorrelation(
            response.getPatient())) 
        {
            status = createPatientNotPendingUncorrelationResponse();
        } else {
            InPersonAuthentication ipa = getIPA(response.getPatient());
            if (ipa == null) {
                status = createPatientIsNotPendingAuthenticationStatus();
            } else {
                String error = getError(status);
                handleMessages(getPatientService().finalizeUncorrelation(
                    response.getPatient(), error));
                handleMessages(getIPAService().finalizeUnauthentication(ipa, 
                    error));
                status = RegistryInterfaceUtils.createOkStatus();
            }
        }
        
        logStatus(operation, patient, status);
        return status;
    }

    /* (non-Javadoc)
     * @see gov.va.med.mhv.integration.registry.services.ejb.
     * PatientIdentityServiceRemote#updateFacilities(
     * gov.va.med.mhv.integration.registry.transfer.Patient, 
     * java.util.ArrayList)
     */
    public Status updateFacilities(Patient patient, 
        ArrayList<Facility> facilities) 
    {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotEmpty("patient.getIcn()", patient.getIcn());
        Precondition.assertNotNull("facilities", facilities);
        final String operation = "Update facilities"; 
        if (LOG.isDebugEnabled()) {
            LOG.debug(operation + " of "+ describe(patient) + " to " 
                + describe(facilities));
        }
        Status status = null;
        PatientServiceResponse response = getPatientByIcn(patient.getIcn());
        if (response.getPatient() != null) {
            RegistryInterfaceUtils.setFacilities(response.getPatient(), 
                facilities);
            //Jazz Enhancement # 29539: Instead of unknown exception, more descriptions are added for the production support.
            status = updatePatient("MFN_M05 Patient location master file Process:Updating Patient with ICN:" + 
            	response.getPatient().getIcn()+" with updating: " + describe(facilities) + ". ", response.getPatient());
        } else {
            status = createPatientDoesNotExistStatus();
        }

        logStatus(operation, patient, status);
        
        return status;
    }


    /* (non-Javadoc)
     * @see gov.va.med.mhv.integration.registry.services.ejb.
     * PatientIdentityServiceRemote#updatePersonalInfo(
     * gov.va.med.mhv.integration.registry.transfer.Patient, 
     * gov.va.med.mhv.integration.registry.transfer.PersonalInfo)
     */
    public Status updatePersonalInfo(Patient patient, PersonalInfo personalInfo)  
    {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotEmpty("patient.getIcn()", patient.getIcn());
        Precondition.assertNotNull("personalInfo", personalInfo);
        final String operation = "Update personal info"; 
        if (LOG.isDebugEnabled()) {
            LOG.debug(operation + " of " + describe(patient) + " to " 
                + describe(personalInfo));
        }
        
        PatientServiceResponse response = getPatientByIcn(patient.getIcn());

        //JAZZ Defect# 20309: If there four or more traits update, DO NOT Update the user and return reject to MVI
        if(isFourOrMoreTraitsUpdate(response.getPatient(), personalInfo)) {
        	UserProfile userProfile = response.getPatient().getUserProfile();
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.updatePersonalInfo() for four or more traits update from MVI for user:" + userProfile.getId() + ":Sending Reject back to MVI");
            }
            
            UserProfileDeactivationReason reason = 
            	getDeactReason(UserProfileDeactivationUtils.A31_POSSIBLE_ICN_MISMATCH_DO_NOT_REACTIVATE_NAME);
            response = deactivatePatientForMultiTraitUpdateError(
            	response.getPatient(), "", reason, 
            	"A31 possible ICN Mismatch (do not reactivate)");
        	return RegistryInterfaceUtils.createErrorStatus("Four or more traits update from MVI");
        }
        
        Status status = null;
        if (response.getPatient() != null) {
            RegistryInterfaceUtils.setPersonalInfo(response.getPatient(), 
                personalInfo);
            //Jazz Enhancement # 29539: Instead of unknown exception, more descriptions are added for the production support. This is for facility update.
            status = updatePatient("ADT_A31 Update person information Process for patient icn:" + response.getPatient().getIcn(), response.getPatient());
        } else {
            status = createPatientDoesNotExistStatus();
        }
        logStatus(operation, patient, status);
        return status;
    }

    public UserProfileDeactivationReason getDeactReason(String name) {
    	return getUserProfileService().findDeactivationReason(name).getUserProfileDeactivationReason();
    }
    
    //JAZZ # 128617: MVI A43 Move message processing.
    public Status deactivatePatient(Patient pat, String referenceId, String reason, String deactivationNote) {
        PatientServiceResponse response = getPatientByIcn(pat.getIcn());
        gov.va.med.mhv.usermgmt.transfer.Patient patient = response.getPatient();
        UserProfileDeactivationReason deactReason = getDeactReason(reason);
		deactivatePatientForMultiTraitUpdateError(patient, referenceId, deactReason, deactivationNote);
		return RegistryInterfaceUtils.createOkStatus();
    }
    
    //JAZZ # 128617: MVI A43 Move message processing.
    public Status deactivatePatientOld(Patient pat, String referenceId, String reason, String deactivationNote) {
        Precondition.assertNotNull("patient", pat);
        Precondition.assertNotEmpty("patient.getIcn()", pat.getIcn());
        final String operation = "Deactivate Patient"; 
        if (LOG.isDebugEnabled()) {
            LOG.debug(operation + " of " + describe(pat));
        }

System.out.println("****************** Ken So ****************** PatientIdentityServiceImpl.deactivatePatient()");    	
        
        PatientServiceResponse response = getPatientByIcn(pat.getIcn());
    	
        gov.va.med.mhv.usermgmt.transfer.Patient patient = response.getPatient();
        
        if(patient == null) {
        	return RegistryInterfaceUtils.createErrorStatus("Patient Not Found");
        }
    	
    	InPersonAuthentication ipa = 
    		getIPAService().getAuthenticationForPatient(patient.getId()).getInPersonAuthentication();
        if (LOG.isDebugEnabled()) {
            LOG.debug("PatientIdentityService.deactivatePatient()unauthenticate for user:" + patient.getUserProfile().getId());
        }
        
		if(ipa.getStatus() == AuthenticationStatus.getEnum(AuthenticationStatus.AUTHENTICATED)) {
            InPersonAuthenticationServiceResponse ipaServiceResponse = getIPAService().unauthenticate(ipa);
            
            if(ipaServiceResponse.getInPersonAuthentication().getStatus() != AuthenticationStatus.getEnum(AuthenticationStatus.UNAUTHENTICATED)) {
            	Auditor.auditMviUnauthenticateEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, false);
            } else {
System.out.println("****************** Ken So ****************** PatientIdentityService.deactivatePatient()unauthenticate for user:" + patient.getUserProfile().getId() + ":SUCCESS");    	
            	Auditor.auditMviUnauthenticateEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, true);
            }
		}
        
    	if(patient.getCorrelationStatus() == PatientCorrelationStatus.getEnum(PatientCorrelationStatus.CORRELATED)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.deactivatePatient()uncorrelate for user:" + patient.getUserProfile().getId());
            }
    		PatientServiceResponse mviSyncServiceResponse = 
    			ServiceFactory.createMviIntegrationService().deletePersonFromCorrelationForHelpDesk(
    					patient, patient.getUserProfile(), "SYSTEM");
            if(mviSyncServiceResponse.getPatient().getCorrelationStatus() !=  PatientCorrelationStatus.getEnum(PatientCorrelationStatus.UNCORRELATED)) {
            	Auditor.auditMviCorrelateEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, false);
            } else {
            	Auditor.auditMviCorrelateEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, true);
System.out.println("****************** Ken So ****************** PatientIdentityService.deactivatePatient()uncorrelate for user:" + patient.getUserProfile().getId() + ":SUCCESS");    	
            }
    	}
    	
        if (LOG.isDebugEnabled()) {
            LOG.debug("PatientIdentityService.deactivatePatient()deactivate for user:" + patient.getUserProfile().getId());
        }
        
        UserProfileDeactivationReason deactReason = getDeactReason(reason);
System.out.println("****************** Ken So ****************** PatientIdentityService.deactivatePatient()UserProfileDeactivationUtils.findReason user:" + patient.getUserProfile().getId() + ":deactReason:" +deactReason.getName());    	
        
    	PatientServiceResponse pPesponse = 
    		getPatientService().deactivatePatientForMultiTraitUpdateError(
    			patient, referenceId, ActivityActorTypeEnumeration.SYSTEM, deactReason, deactivationNote);
    	patient = pPesponse.getPatient();
        if(pPesponse.getMessages().hasErrorMessages()) {
     	   Auditor.auditAccountDeactivateEvent(
     			   patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, false);
     	   return RegistryInterfaceUtils.createErrorStatus("Error in Patient Deactivation"); 
        } else {
System.out.println("****************** Ken So ****************** PatientService().deactivatePatientForMultiTraitUpdateError user:" + patient.getUserProfile().getId() + ":SUCCESS");    	
System.out.println("****************** Ken So ****************** PatientService().deactivatePatientForMultiTraitUpdateError lastName:" + patient.getUserProfile().getLastName());    	
     	   Auditor.auditAccountDeactivateEvent(
     			  patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, true);
        }
        
        return RegistryInterfaceUtils.createOkStatus();
    }
    
    
    //JAZZ Defect# 20309: If there four or more traits update, DO NOT Update the user and deactivate the user.
    public PatientServiceResponse deactivatePatientForMultiTraitUpdateError(gov.va.med.mhv.usermgmt.transfer.Patient patient,
        String referenceId, UserProfileDeactivationReason reason, String deactivationNote) {
    	InPersonAuthentication ipa = 
    		getIPAService().getAuthenticationForPatient(patient.getId()).getInPersonAuthentication();
        if (LOG.isDebugEnabled()) {
            LOG.debug("PatientIdentityService.deactivatePatientForMultiTraitUpdateError()unauthenticate for user:" + patient.getUserProfile().getId());
        }
        
		if(ipa.getStatus() == AuthenticationStatus.getEnum(AuthenticationStatus.AUTHENTICATED)) {
            InPersonAuthenticationServiceResponse ipaServiceResponse = getIPAService().unauthenticate(ipa);
            
            if(ipaServiceResponse.getInPersonAuthentication().getStatus() != AuthenticationStatus.getEnum(AuthenticationStatus.UNAUTHENTICATED)) {
            	Auditor.auditMviUnauthenticateRequestEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, false);
            } else {
            	Auditor.auditMviUnauthenticateRequestEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, true);
            }
		}
        
    	if(patient.getCorrelationStatus() == PatientCorrelationStatus.getEnum(PatientCorrelationStatus.CORRELATED)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.deactivatePatientForMultiTraitUpdateError()uncorrelate for user:" + patient.getUserProfile().getId());
            }
    		PatientServiceResponse mviSyncServiceResponse = 
    			ServiceFactory.createMviIntegrationService().deletePersonFromCorrelationForHelpDesk(
    					patient, patient.getUserProfile(), "");
            if(mviSyncServiceResponse.getPatient().getCorrelationStatus() !=  PatientCorrelationStatus.getEnum(PatientCorrelationStatus.UNCORRELATED)) {
            	Auditor.auditMviCorrelateEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, false);
            } else {
            	Auditor.auditMviCorrelateEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, true);
            }
    	}
    	
        if (LOG.isDebugEnabled()) {
            LOG.debug("PatientIdentityService.deactivatePatientForMultiTraitUpdateError()deactivate for user:" + patient.getUserProfile().getId());
        }
        
    	PatientServiceResponse response = 
    		getPatientService().deactivatePatientForMultiTraitUpdateError(
    			patient, referenceId, ActivityActorTypeEnumeration.SYSTEM, reason, deactivationNote);
        if(response.getMessages().hasErrorMessages()) {
     	   Auditor.auditAccountDeactivateEvent(
     			   patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, false);
        } else {
     	   Auditor.auditAccountDeactivateEvent(
     			  patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, true);
        }
        
        return response;
    }
    
    public PatientServiceResponse deactivatePatientForAutmatedDeathNotification(gov.va.med.mhv.usermgmt.transfer.Patient patient,
        String referenceId, UserProfileDeactivationReason reason, String deactivationNote, Date dateOfDeath) {
    	InPersonAuthentication ipa = 
    		getIPAService().getAuthenticationForPatient(patient.getId()).getInPersonAuthentication();
        if (LOG.isDebugEnabled()) {
            LOG.debug("PatientIdentityService.deactivatePatientForAutmatedDeathNotification()unauthenticate for user:" + patient.getUserProfile().getId());
        }
        
		if(ipa.getStatus() == AuthenticationStatus.getEnum(AuthenticationStatus.AUTHENTICATED)) {
            InPersonAuthenticationServiceResponse ipaServiceResponse = getIPAService().unauthenticate(ipa);
            
            if(ipaServiceResponse.getInPersonAuthentication().getStatus() != AuthenticationStatus.getEnum(AuthenticationStatus.UNAUTHENTICATED)) {
            	Auditor.auditMviUnauthenticateRequestEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, false);
            } else {
            	Auditor.auditMviUnauthenticateRequestEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, true);
            }
		}
        
    	if(patient.getCorrelationStatus() == PatientCorrelationStatus.getEnum(PatientCorrelationStatus.CORRELATED)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.deactivatePatientForAutmatedDeathNotification()uncorrelate for user:" + patient.getUserProfile().getId());
            }
    		PatientServiceResponse mviSyncServiceResponse = 
    			ServiceFactory.createMviIntegrationService().deletePersonFromCorrelationForHelpDesk(
    					patient, patient.getUserProfile(), "");
            if(mviSyncServiceResponse.getPatient().getCorrelationStatus() !=  PatientCorrelationStatus.getEnum(PatientCorrelationStatus.UNCORRELATED)) {
            	Auditor.auditMviCorrelateEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, false);
            } else {
            	Auditor.auditMviCorrelateEvent(patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, true);
            }
    	}
    	
        if (LOG.isDebugEnabled()) {
            LOG.debug("PatientIdentityService.deactivatePatientForAutmatedDeathNotification()deactivate for user:" + patient.getUserProfile().getId());
        }
        
    	PatientServiceResponse response = 
    		getPatientService().deactivatePatientInHelpDesk(
    			patient, referenceId, ActivityActorTypeEnumeration.SYSTEM, reason, deactivationNote, dateOfDeath);
    	
        if(response.getMessages().hasErrorMessages()) {
     	   Auditor.auditAccountDeactivateEvent(
     			   patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, false);
        } else {
     	   Auditor.auditAccountDeactivateEvent(
     			  patient.getUserProfile().getId(), ActivityActorTypeEnumeration.SYSTEM, true);
        }
        
        return response;
    }

    private boolean isFourOrMoreTraitsUpdate(gov.va.med.mhv.usermgmt.transfer.Patient patient, PersonalInfo personalInfo) {
    	int traitUpdateCount = 0;
    	
        UserProfile userProfile = patient.getUserProfile();
        
        if (LOG.isDebugEnabled()) {
            LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() for user profile id:" + userProfile.getId());
        }
        
        //If a trait is not the same, we increment the count. This applies to first name, middle name, last name, ssn, gender and DOB.
        if(!userProfile.getFirstName().equalsIgnoreCase(personalInfo.getName().getFirstName())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() first name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        //Middle name can be null on both sides so we need to cover all permutations.
        if(userProfile.getMiddleName()==null && personalInfo.getName().getMiddleName()!=null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() middle name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        } else if(userProfile.getMiddleName()!=null && personalInfo.getName().getMiddleName()==null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() middle name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        } else if(userProfile.getMiddleName()==null && personalInfo.getName().getMiddleName()==null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() no middle name for user profile id:" + userProfile.getId());
            }
        } else if(!userProfile.getMiddleName().equalsIgnoreCase(personalInfo.getName().getMiddleName())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() middle name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        if(!userProfile.getLastName().equalsIgnoreCase(personalInfo.getName().getLastName())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() last name is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        if(!userProfile.getBirthDate().equals(personalInfo.getDateOfBirth())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() birth date is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        if(!userProfile.getGender().getName().equalsIgnoreCase(personalInfo.getGender().getName())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() gender is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        String ssn = userProfile.getSsn().replaceAll("-", "");
        if(!ssn.equals(personalInfo.getSsn())) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("PatientIdentityService.isFourOrMoreTraitsUpdate() ssn is different for user profile id:" + userProfile.getId());
            }
        	traitUpdateCount++;
        }
        
        if(traitUpdateCount>3) {
        	return true;
        }
        
        return false;
    }
    
    /* (non-Javadoc)
     * @see gov.va.med.mhv.integration.registry.services.ejb.
     * PatientIdentityServiceRemote#changeIcn(
     * gov.va.med.mhv.integration.registry.transfer.Patient, java.lang.String)
     */
    public Status changeIcn(Patient patient, String icn) {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotEmpty("patient.getIcn()", patient.getIcn());
        Precondition.assertNotEmpty("icn", icn);
        final String operation = "Update icn"; 
        if (LOG.isDebugEnabled()) {
            LOG.debug(operation + " of " + describe(patient) + " to'" + icn 
                + "'");
        }
        Status status = null;
        PatientServiceResponse response = getPatientByIcn(patient.getIcn());
        if (response.getPatient() != null) {
            response.getPatient().setIcn(icn);
            status = updatePatient(response.getPatient());
        } else {
            status = createPatientDoesNotExistStatus();
        }
        logStatus(operation, patient, status);
        return status;
    }




    /* (non-Javadoc)
     * @see gov.va.med.mhv.integration.registry.services.ejb.
     * PatientIdentityServiceRemote#removeCorrelation(
     * gov.va.med.mhv.integration.registry.transfer.Patient)
     */
    public Status removeCorrelation(Patient patient) {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotEmpty("patient.getIcn()", patient.getIcn());
        final String operation = "Remove correlation";
        if (LOG.isDebugEnabled()) {
            LOG.debug(operation + " of " + describe(patient));
        }
        Status status = null;
        PatientServiceResponse response = getPatientByIcn(patient.getIcn());
        if (response.getPatient() == null) {
            status = createPatientDoesNotExistStatus();
        } else if (PatientCorrelationStatusUtils.isPendingCorrelation(response.
            getPatient())) 
        {
            status = finalizeCorrelation(patient, new Status(StatusType.
                Error, MessageKeys.PATIENT_MPI_CORRELATION_FAILED));
        } else if (PatientCorrelationStatusUtils.isPendingUncorrelation(
            response.getPatient())) 
        {
            status = finalizeUncorrelation(patient, new Status(StatusType.
                Error, MessageKeys.PATIENT_MPI_UNCORRELATION_FAILED));
        } else {
            handleMessages(getPatientService().removeCorrelation(response.
                getPatient()));
            InPersonAuthentication ipa = getIPA(response.getPatient());
            if (ipa != null) {
                handleMessages(getIPAService().terminateAuthentication(ipa));
            }
            status = RegistryInterfaceUtils.createOkStatus();
        }
        logStatus(operation, patient, status);
        return status;
    }

    private void handleMessages(String description, EntityServiceResponse response) {
        if (response == null) {
            return;
        }
        handleMessages(description, (ServiceResponse) response);
        if (response.getEntity() != null) {
            MessagesSet messages = response.getEntity().getAllMessages(); 
            if (isEmpty(messages)) {
                return;
            }

            logMessages(messages);
            if (messages.hasErrorMessages()) {
            	StringBuffer buffer = new StringBuffer();
            	buffer.append(new MessagesStringBuilder().append(messages, getClass()).getErrorString());
                //Jazz Enhancement # 29539: Instead of unknown exception, more descriptions are added for the production support.
            	throw new RuntimeException(description + ": " + buffer.toString());
            	//This is an old code that didn't have any description. Left it commented for research purpse            	
            	//throw new RuntimeException(MessageKeys.UNKNOWN_EXCEPTION_OCCURRED);
            }
        }
    }


    /* (non-Javadoc)
     * @see gov.va.med.mhv.integration.registry.services.ejb.
     * PatientIdentityServiceRemote#mergePatients(
     * gov.va.med.mhv.integration.registry.transfer.Patient, 
     * gov.va.med.mhv.integration.registry.transfer.Patient, java.lang.String)
     */
    public Status mergePatients(Patient fromPatient, Patient toPatient, 
        String referenceId) 
    {
        Precondition.assertNotNull("fromPatient", fromPatient);
        Precondition.assertNotNull("toPatient", toPatient);
        Precondition.assertNotBlank("referenceId", referenceId);
        
        final String operation = "Merge ";
        if (LOG.isDebugEnabled()) {
            LOG.debug(operation + describe(fromPatient) + " and " +
                describe(toPatient) + " with reference id '" + referenceId 
                + "'");
        }
        Status status = null;
        PatientServiceResponse response = getPatientByIcn(fromPatient.getIcn());
        //Jazz Enhancement # 29539: Instead of unknown exception, more descriptions are added for the production support.
        handleMessages("ADT-A24 Link Patient Information Process:Getting From Patient with ICN:" + fromPatient.getIcn(), response);

        PatientServiceResponse toResponse = getPatientByIcn(toPatient.getIcn());
        //Jazz Enhancement # 29539: Instead of unknown exception, more descriptions are added for the production support.
        handleMessages("ADT-A24 Link Patient Information Process:Getting To Patient with ICN:" + toPatient.getIcn(), toResponse);
        if ((response.getPatient() == null) 
            || (toResponse.getPatient() == null))
        {
            status = createPatientDoesNotExistStatus();
        } else {
        	UserProfileDeactivationReason reason = 
        		UserProfileDeactivationUtils.findReason(UserProfileDeactivationUtils.MPI_PATIENT_MERGE_NAME);
        	
            response = getPatientService().invalidatePatient(response.getPatient(), referenceId, reason);
            //Jazz Enhancement # 29539: Instead of unknown exception, more descriptions are added for the production support.
            handleMessages("ADT-A24 Link Patient Information Process:invalidating Patient with ICN:" + 
            	response.getPatient().getIcn(), toResponse);
            sendMergeAlert(response.getPatient(), toResponse.getPatient());
            status = RegistryInterfaceUtils.createOkStatus();
        }
        logStatus(operation, fromPatient, toPatient, status);
        return status;
    }
    
    private void sendMergeAlert(
        gov.va.med.mhv.usermgmt.transfer.Patient fromPatient, 
        gov.va.med.mhv.usermgmt.transfer.Patient toPatient) 
    {
        HelpDeskAlert alert = HelpDeskAlertProperties.getInstance().
            createMergePatientsAlert();
        alert.formatSubject(fromPatient.getInvalidationReferenceId());
        alert.formatMessage(fromPatient.getInvalidatedIcn(), toPatient.getIcn(), 
            Calendar.getInstance(), fromPatient.getUserProfile().getMhvId(), 
            toPatient.getUserProfile().getMhvId(), fromPatient.
            getInvalidationReferenceId());
        VoidServiceResponse response = alert.send();
        if (ServiceResponseUtils.hasErrorMessages(response)) {
            // TODO: Do we nee to have a try-catch block as well?
            // Just log the messages
            // We should not fail the merge request because we cannot send 
            // the email
            ServiceResponseUtils.logMessages(response, getClass(), LOG);
        }
    }

    private PatientServiceResponse getPatient(Patient patient) {
        assert patient != null;
        PatientServiceResponse response = null;
        if (!StringUtils.isBlank(patient.getIcn())) {
            response = getPatientByIcn(patient.getIcn());
        } else {
            List userProfileBOs = UserProfileBO.
                getProfileByNameBirthDateSsn(
                    patient.getPersonalInfo().getName().getFirstName(), 
                    patient.getPersonalInfo().getName().getLastName(),
                    patient.getPersonalInfo().getDateOfBirth(),
                    patient.getPersonalInfo().getSsn());
            if ((userProfileBOs != null) && (userProfileBOs.size() > 0)) {
                response = getPatientService().getPatientForUser(
                    ((UserProfileBO) userProfileBOs.iterator().next()).
                    getUserProfileValues());
            } else {
                response = new PatientServiceResponse();
            }
        }
        return response;
        
    }

    private PatientServiceResponse getPatientByIcn(String icn) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("getPatientByIcn(" + icn + ")");
        }
        PatientServiceResponse response = getPatientService().
            getPatientByIcn(icn);
        handleMessages(response);
        return response;
    }

    private UserProfileService getUserProfileService() {
        return ServiceFactory.createUserProfileService();
    }   

    private PatientService getPatientService() {
        return ServiceFactory.createPatientService();
    }   

    private InPersonAuthenticationService getIPAService() {
        return ServiceFactory.createInPersonAuthenticationService();
    }   

    //Jazz Enhancement # 29539: Instead of unknown exception, more descriptions are added for the production support.
    private Status updatePatient(String description, 
        gov.va.med.mhv.usermgmt.transfer.Patient patient) {
        PatientServiceResponse response = getPatientService().
            updatePatientRegistryInformation(patient);
        
        MessagesSet messages = response.getEntity().getAllMessages(); 
        if (messages.hasErrorMessages()) {
            //Jazz Enhancement # 29539: Find out which facilities do not exist in MHV but in MVI.
			boolean facilityNotFound = false;
        	for (Object o: patient.getFacilitys()) {
        		gov.va.med.mhv.usermgmt.transfer.Facility facility = (gov.va.med.mhv.usermgmt.transfer.Facility)o;
        		List existingFacilities = FacilityInfoBO.queryByStationNumber(facility.getName());
				if(existingFacilities == null || existingFacilities.size() == 0){
					if(!facilityNotFound) {
						description = description + "The following facilities are not found in MHV {"; 
						facilityNotFound = true;
					} else {
						description = description + " ";
					}
					description = description + facility.getName();
				}
				description = description + "}";
			}
        }	
        
        handleMessages(description, response);
        return RegistryInterfaceUtils.createOkStatus();
    }
    
    private Status updatePatient(
            gov.va.med.mhv.usermgmt.transfer.Patient patient) 
    {
    	return updatePatient(null, patient);
    }
    
    private void handleMessages(ServiceResponse response) {
    	handleMessages(null, response);
    }
    
    private void handleMessages(String description, ServiceResponse response) {
        if (response == null) {
            return;
        }
        if (isEmpty(response.getMessages())) {
            return;
        }
        logMessages(response.getMessages());
        if (response.getMessages().hasErrorMessages()) {
        	StringBuffer buffer = new StringBuffer();
        	buffer.append(new MessagesStringBuilder().append(response.getMessages(), getClass()).getErrorString());
            //Jazz Enhancement # 29539: Instead of unknown exception, more descriptions are added for the production support.
        	throw new RuntimeException(description + ": " + buffer.toString());
        	//This is an old code that didn't have any description. Left it commented for research purpse            	
        	//throw new RuntimeException(MessageKeys.UNKNOWN_EXCEPTION_OCCURRED);
        }
    }

    private void logMessages(MessagesSet messages) {
        Precondition.assertNotNull("messages", messages);
        if (isEmpty(messages)) {
            return;
        }
        MessagesStringBuilder builder = new MessagesStringBuilder(); 
        builder.append(messages, getClass());
        if (messages.hasInformationalMessages()) {
            if (LOG.isInfoEnabled()) {
                LOG.info(builder.getInfoString());
            }
        }
        if (messages.hasErrorMessages()) {
            LOG.error(builder.getErrorString());
        }
    }

    private boolean isEmpty(MessagesSet messages) {
        return (messages == null) 
            || (messages.getErrorMessageCount() + 
                messages.getInformationalMessageCount() == 0);
    }

    private void logStatus(String description, Patient patient, Status status) {
        logStatus(description, patient, null, status);
    }

    private void logStatus(String description, Patient fromPatient, 
        Patient toPatient, Status status) 
    {
        Precondition.assertNotNull("status", status);
        if (StatusType.Error.equals(status.getStatus())) {
            LOG.error(description + " of " + describe(fromPatient) +
                ((toPatient != null) ? " and " + describe(toPatient) : "") +
                " failed, because " + status.getStatusDescription());
        } else if (LOG.isInfoEnabled()) {
            LOG.info(description + " of " + describe(fromPatient) + 
                ((toPatient != null) ? " and " + describe(toPatient) : "") +
                " succeeded");
        }
    }

    private String describe(Patient patient) {
        return RegistryInterfaceUtils.describe(patient);
    }

    private String describe(PersonalInfo personalInfo) {
        return RegistryInterfaceUtils.describe(personalInfo);
    }

    private String describe(List<Facility> facilities) {
        return RegistryInterfaceUtils.describe(facilities);
    }

    private String describe(Status status) {
        return RegistryInterfaceUtils.describe(status);
    }

    private static Status createPatientDoesNotExistStatus() {
        return RegistryInterfaceUtils.createErrorStatus(MessageKeys.
            PATIENT_NOT_FOUND);
    }

    private static Status createPatientNotPendingCorrelationResponse() {
        return RegistryInterfaceUtils.createErrorStatus(MessageKeys.
            PATIENT_IS_NOT_PENDING_CORRELATION);
    }

    private static Status createPatientNotPendingUncorrelationResponse() {
        return RegistryInterfaceUtils.createErrorStatus(MessageKeys.
            PATIENT_IS_NOT_PENDING_UNCORRELATION);
    }

    private static Status createPatientIsNotPendingAuthenticationStatus() {
        return RegistryInterfaceUtils.createErrorStatus(MessageKeys.
            PATIENT_IS_NOT_PENDING_AUTHENTICATION);
    }

    private static Status createPatientIsNotPendingSynchronizationStatus()  {
        return RegistryInterfaceUtils.createErrorStatus(MessageKeys.
            PATIENT_IS_NOT_PENDING_SYNCHRONIZATION);
    }

    private String getError(Status status) {
        return ((status != null) && StatusType.Error.equals(status.getStatus()))
            ? status.getStatusDescription() : null;
    }
}
